/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim: set ts=8 sts=2 et sw=2 tw=80: */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/. */

#ifndef mozilla_dom_WebIDLGlobalNameHash_h__
#define mozilla_dom_WebIDLGlobalNameHash_h__

#include "js/RootingAPI.h"
#include "nsTArray.h"
#include "mozilla/dom/BindingDeclarations.h"

class JSLinearString;

namespace mozilla::dom {

enum class BindingNamesOffset : uint16_t;

namespace constructors::id {
enum ID : uint16_t;
}  // namespace constructors::id

struct WebIDLNameTableEntry {
  // Check whether a constructor should be enabled for the given object.
  // Note that the object should NOT be an Xray, since Xrays will end up
  // defining constructors on the underlying object.
  using ConstructorEnabled = bool (*)(JSContext* cx, JS::Handle<JSObject*> obj);

  BindingNamesOffset mNameOffset;
  uint16_t mNameLength;
  constructors::id::ID mConstructorId;
  CreateInterfaceObjectsMethod mCreate;
  // May be null if enabled unconditionally
  ConstructorEnabled mEnabled;
};

class WebIDLGlobalNameHash {
 public:
  using ConstructorEnabled = WebIDLNameTableEntry::ConstructorEnabled;

  // Returns false if something failed. aFound is set to true if the name is in
  // the hash, whether it's enabled or not.
  static bool DefineIfEnabled(
      JSContext* aCx, JS::Handle<JSObject*> aObj, JS::Handle<jsid> aId,
      JS::MutableHandle<mozilla::Maybe<JS::PropertyDescriptor>> aDesc,
      bool* aFound);

  static bool MayResolve(jsid aId);

  // The type of names we're asking for.
  enum NameType {
    // All WebIDL names enabled for aObj.
    AllNames,
    // Only the names that are enabled for aObj and have not been resolved for
    // aObj in the past (and therefore can't have been deleted).
    UnresolvedNamesOnly
  };
  // Returns false if an exception has been thrown on aCx.
  static bool GetNames(JSContext* aCx, JS::Handle<JSObject*> aObj,
                       NameType aNameType,
                       JS::MutableHandleVector<jsid> aNames);

  // Helpers for resolving & enumerating names on the system global.
  // NOTE: These are distinct as it currently lacks a ProtoAndIfaceCache, and is
  // an XPCOM global.
  static bool ResolveForSystemGlobal(JSContext* aCx, JS::Handle<JSObject*> aObj,
                                     JS::Handle<jsid> aId, bool* aResolvedp);

  static bool NewEnumerateSystemGlobal(
      JSContext* aCx, JS::Handle<JSObject*> aObj,
      JS::MutableHandleVector<jsid> aProperties, bool aEnumerableOnly);

 private:
  friend struct WebIDLNameTableEntry;

  // Look up an entry by key name. `nullptr` if the entry was not found.
  // The impl of GetEntry is generated by Codegen.py in RegisterBindings.cpp
  static const WebIDLNameTableEntry* GetEntry(JSLinearString* aKey);

  // The total number of names in the hash.
  // The value of sCount is generated by Codegen.py in RegisterBindings.cpp.
  static const uint32_t sCount;

  // The name table entries in the hash.
  // The value of sEntries is generated by Codegen.py in RegisterBindings.cpp.
  static const WebIDLNameTableEntry sEntries[];
};

}  // namespace mozilla::dom

#endif  // mozilla_dom_WebIDLGlobalNameHash_h__
